<?php

/**
 * Sets up theme defaults and registers support for various WordPress features.
 *
 * Note that this function is hooked into the after_setup_theme hook, which
 * runs before the init hook. The init hook is too late for some features, such
 * as indicating support for post thumbnails.
 */
function che_general_setup() {
	/*
	 * Make theme available for translation.
	 * Translations can be filed in the /languages/ directory.
	 * If you're building a theme based on Start-theme, use a find and replace
	 * to change 'general' to the name of your theme in all the template files.
	 */
	load_theme_textdomain( 'general', get_template_directory() . '/languages' );

	// Add default posts and comments RSS feed links to head.
	add_theme_support( 'automatic-feed-links' );

	/*
	 * Let WordPress manage the document title.
	 * By adding theme support, we declare that this theme does not use a
	 * hard-coded <title> tag in the document head, and expect WordPress to
	 * provide it for us.
	 */
	add_theme_support( 'title-tag' );

	/*
	 * Enable support for Post Thumbnails on posts and pages.
	 *
	 * @link https://developer.wordpress.org/themes/functionality/featured-images-post-thumbnails/
	 */
	add_theme_support( 'post-thumbnails' );

	/*
	 * Switch default core markup for search form, comment form, and comments
	 * to output valid HTML5.
	 */
	add_theme_support(
		'html5',
		[
			'search-form',
			'comment-form',
			'comment-list',
			'gallery',
			'caption',
		]
	);

	// Add theme support for selective refresh for widgets.
	add_theme_support( 'customize-selective-refresh-widgets' );

	/**
	 * By default, you can align blocks to the left, right or center. This adds two additional
	 * options – wide and full – to the editor block controls. To utilize them on the site
	 * frontend you need to add corresponding styles for these new generated classes.
	 *
	 * [ .alignleft | .aligncenter | .alignright | .alignwide | .alignfull ]
	 *
	 * @link https://www.billerickson.net/full-and-wide-alignment-in-gutenberg/
	 */
	add_theme_support( 'align-wide' );

	/**
	 * When editing paragraph text, a user can select different sizes in the settings panel.
	 * This lets you define what sizes should be available, adds a CSS class like
	 * .has-{size-slug}-font-size.
	 */
	add_theme_support( 'editor-font-sizes',
		che_general_config( 'editor_font_sizes' )
	);

	/**
	 * And disable settings any custom font size too.
	 */
	add_theme_support( 'disable-custom-font-sizes' );

	/**
	 * Define a few brand colors that make a custom color palette. This adds a CSS class like
	 * .has-{color-slug}-background-color and .has-{color-slug}-color.
	 */
	add_theme_support( 'editor-color-palette',
		che_get_theme_colors( )
	);

	/**
	 * And disable color picker.
	 */
	add_theme_support( 'disable-custom-colors' );

	/**
	 * Configure intermediate image sizes.
	 */
	$images_sizes = che_general_config( 'image_sizes' );
	if ( $images_sizes && is_array( $images_sizes ) ) {
		foreach ( $images_sizes as $item ) {
			if ( ! empty( $item['width'] ) && ! empty( $item['height'] ) ) {
				add_image_size( $item['name'],
					(int) $item['width'],
					(int) $item['height'],
					( $item['crop'] ?: false ) );
			}
		}
	}

	/**
	 * Registering menus
	 */
	register_nav_menus( che_general_config( 'register_nav_menus' ) );
}

add_action( 'after_setup_theme', 'che_general_setup' );

/**
 * Adjusts WordPress image quality.
 */
add_filter( 'jpeg_quality', static function () {
	return che_general_config( 'jpeg_quality' );
} );

/**
 * Custom Excerpts for posts and pages.
 *
 * @package CHE
 */
add_filter( 'excerpt_length', function ( $length ) {
	return che_general_config( 'excerpt_length' );
}, 999 );

/**
 * Custom Excerpt more for posts and pages
 *
 * @param $more
 *
 * @return mixed|null
 */
add_filter( 'excerpt_more', function ( $more ) {
	return che_general_config( 'excerpt_more' );
} );

/**
 * Disable the emoji support.
 */
function che_general_disable_emojis() {
	remove_action( 'wp_head', 'print_emoji_detection_script', 7 );
	remove_action( 'admin_print_scripts', 'print_emoji_detection_script' );
	remove_action( 'wp_print_styles', 'print_emoji_styles' );
	remove_action( 'admin_print_styles', 'print_emoji_styles' );
	remove_filter( 'the_content_feed', 'wp_staticize_emoji' );
	remove_filter( 'comment_text_rss', 'wp_staticize_emoji' );
	remove_filter( 'wp_mail', 'wp_staticize_emoji_for_email' );

	add_filter( 'tiny_mce_plugins', 'che_general_disable_emojis_tinymce' );
	add_filter( 'wp_resource_hints', 'che_general_disable_emojis_remove_dns_prefetch', 10, 2 );
}

add_action( 'init', 'che_general_disable_emojis' );

/**
 * Filter function used to remove the TinyMCE emoji plugin.
 *
 * @param array $plugins
 *
 * @return array Difference between the two arrays
 */
function che_general_disable_emojis_tinymce( $plugins ) {
	if ( is_array( $plugins ) ) {
		return array_diff( $plugins, [ 'wpemoji' ] );
	} else {
		return [];
	}
}

/**
 * Remove emoji CDN hostname prefetch from resource hints.
 *
 * @param array $urls URLs to print for resource hints.
 * @param string $relation_type The relation type the URLs are printed for.
 *
 * @return array Difference between the two arrays.
 */
function che_general_disable_emojis_remove_dns_prefetch( $urls, $relation_type ) {
	if ( 'dns-prefetch' === $relation_type ) {
		/** This filter is documented in wp-includes/formatting.php */
		$emoji_svg_url = apply_filters( 'emoji_svg_url', 'https://s.w.org/images/core/emoji/2/svg/' );
		$urls          = array_diff( $urls, [ $emoji_svg_url ] );
	}

	return $urls;
}

/**
 * Removes the width and height attributes of <img> tags for SVG
 *
 * Without this filter, the width and height are set to "1" since
 * WordPress core can't seem to figure out an SVG file's dimensions.
 *
 * For SVG:s, returns an array with file url, width and height set
 * to null, and false for 'is_intermediate'.
 *
 * @wp-hook image_downsize
 *
 * @param mixed $out Value to be filtered
 * @param int $id Attachment ID for image.
 *
 * @return bool|array False if not in admin or not SVG. Array otherwise.
 */
function che_general_fix_svg_size_attributes( $out, $id ) {
	$image_url = wp_get_attachment_url( $id );
	$file_ext  = pathinfo( $image_url, PATHINFO_EXTENSION );

	if ( is_admin() || 'svg' !== $file_ext ) {
		return false;
	}

	return [
		$image_url,
		null,
		null,
		false,
	];
}

add_filter( 'image_downsize', 'che_general_fix_svg_size_attributes', 10, 2 );

/**
 * Allow to upload and recognize SVG files as valid file type.
 *
 * @param array $mimes
 *
 * @return array
 */
add_filter( 'upload_mimes', static function ( $mimes ) {
	$mimes['svg'] = 'image/svg+xml';

	return $mimes;
} );

/**
 * Remove previous and next links from head and SQL queries used to generate them.
 */
remove_action( 'wp_head', 'adjacent_posts_rel_link_wp_head' );

/**
 * Add scripts after open tag <body> from admin panel
 */
add_action('wp_body_open', function (){
	if ( function_exists( 'get_field' ) ) {
		echo get_field( 'scripts_body', 'options' );
	}
});

/**
 * Add scripts before close tag </head> from admin panel
 */
add_action('wp_head', function (){
	if ( function_exists( 'get_field' ) ) {
		echo get_field( 'scripts_head', 'options' );
	}
});

/**
 * Add scripts before close </body> from admin panel
 */
add_action('wp_footer', function (){
	if ( function_exists( 'get_field' ) ) {
		echo get_field( 'scripts_footer', 'options' );
	}
});

/**
 * Ensure post-type meta boxes are checked by default in the nav-menus.php screen.
 */
function che_set_metaboxes_checked_by_default(): void {
	// Ensure this runs only on the 'nav-menus' screen
	if (($screen = get_current_screen()) && $screen->id === 'nav-menus') {

		// Get the current user ID and define the option keys
		$user_id = get_current_user_id();
		$user_option_key = 'custom_metaboxhidden_nav_menus_' . $user_id;

		// Check if the user already modified the metabox visibility
		if (get_user_option($user_option_key)) {
			return; // Exit early to prevent unnecessary updates
		}

		// Get all public post types
		$post_types = get_post_types(['public' => true], 'names');

		// Retrieve hidden metaboxes for the nav-menus screen or set to an empty array
		$hidden = get_user_option('metaboxhidden_nav-menus') ?? [];

		if (!is_array($hidden)) {
			$hidden = [];
		}

		// Ensure post type metaboxes are not hidden
		$updated_hidden = array_filter($hidden, function($meta_box_id) use ($post_types) {
			foreach ($post_types as $post_type) {
				if ($meta_box_id === 'add-post-type-' . sanitize_key($post_type)) {
					return false; // Exclude from the new hidden array
				}
			}
			return true; // Keep other hidden items intact
		});

		// Update the user's hidden metaboxes if changes were made
		if ($updated_hidden !== $hidden) {
			update_user_option($user_id, 'metaboxhidden_nav-menus', $updated_hidden, true);
		}

		// Mark the metabox state as modified for this user to avoid repeated updates
		update_user_option($user_id, $user_option_key, true);
	}
}

// Hook the function to the 'current_screen' action
add_action('current_screen', 'che_set_metaboxes_checked_by_default');

/**
 * Retrieve the theme colors from scss file.
 *
 * This function returns an array of colors used in the theme.
 *
 * @return array An associative array of theme colors.
 */
function che_get_theme_colors(): array {
    $default_colors = array(
		array(
			'name'  => __( 'Primary Blue', 'general' ),
			'slug'  => 'primary-blue',
			'color' => '#008dcd',
		),
		array(
			'name'  => __( 'Dark Blue', 'general' ),
			'slug'  => 'dark-blue',
			'color' => '#00355d',
		),
	);
    $source = THEME_FRONTEND_DIR . '/src/styles/common/_root.scss';

    if (!file_exists($source)) {
        return $default_colors;
    }

    $lines = file($source, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
    if ($lines === false) {
        error_log('Error reading file: ' . $source);
        return $default_colors;
    }

	$colors = array();

    foreach ($lines as $line) {
        if ( str_contains( $line, '--color' ) ) {
            $color = explode(':', $line);
            if (count($color) < 2) {
                continue;
            }

            $slug = trim(str_replace('--color-', '', $color[0]));
            $name = ucfirst(trim(str_replace('-', ' ', $slug)));
            $value = trim(str_replace(';', '', $color[1]));

            $colors[] = [
                'name' => sanitize_text_field($name),
                'slug' => sanitize_text_field($slug),
                'color' => sanitize_hex_color($value)
            ];
        }
    }

    return $colors;
}
