<?php
/**
 * ACF Pro Integration.
 *
 * Customise built-in ACF Pro functionality, orchestrate blocks, fields, groups
 * and options pages.
 *
 * @since   1.0.0
 * @package CHE\ACF
 */


/**
 * Adds ACF Pro license key
 *
 * Triggered on license key on theme activation
 */
function che_general_auto_set_license_keys() {
	if ( ! get_option( 'acf_pro_license' ) ) {
		$save = [
			'key' => che_general_config( 'acf_key' ),
			'url' => home_url(),
		];

		$save = maybe_serialize( $save );
		$save = base64_encode( $save );

		update_option( 'acf_pro_license', $save );
	}
}

add_action( 'after_switch_theme', 'che_general_auto_set_license_keys' );

/**
 * Configure Google Maps API.
 *
 * @link https://www.advancedcustomfields.com/resources/acf-fields-google_map-api/
 * @link https://developers.google.com/maps/documentation/javascript/get-api-key
 *
 * @return array
 */
add_filter( 'acf/fields/google_map/api', function ( $args ) {
	$args['key'] = get_field( 'google_maps_api_key', 'options' );

	return $args;
} );


/**
 * Allow only in local development
 *
 * @return bool
 */
add_filter( 'acf/settings/show_admin', function () {
	$environment_type = wp_get_environment_type();
	if ( $environment_type == 'local' ) {
		return true;
	}

	return false;
} );


/**
 * Filter allowed blocks to include only specific categories and exclude specific blocks.
 *
 * @return array List of allowed block types.
 */
function che_filter_allowed_blocks(): array {
	// Get all registered blocks in WordPress.
	$all_blocks = WP_Block_Type_Registry::get_instance()->get_all_registered();

	// Initialize an array to store filtered blocks.
	$filtered_blocks = [];

	// Categories to include in the editor.
	$include_categories = [
		'text',
		'media',
		'design',
		'widgets',
		'sections'
	];

	// Specific blocks to exclude, as marked in the images.
	$exclude_blocks = [
		'core/pullquote',
		'core/freeform',
		'core/archives',
		'core/calendar',
		'core/categories',
		'core/latest-comments',
		'core/latest-posts',
		'core/page-list',
		'core/rss',
		'core/tag-cloud',
		'core/preformatted',
		'core/verse',
	];

	// Loop through each block and filter by categories and exclusions.
	foreach ( $all_blocks as $block_name => $block_data ) {
		if ( in_array( $block_data->category, $include_categories ) && // Check if category is allowed
		     ! in_array( $block_name, $exclude_blocks ) // Check if block is not in the exclusion list
		) {
			$filtered_blocks[] = $block_name;
		}
	}

	// Return the filtered list of block types.
	return $filtered_blocks;
}

add_filter( 'allowed_block_types_all', function ( $allowed_block_types, $block_editor_context ) {
	return che_filter_allowed_blocks();
}, 10, 2 );


/**
 * Fix files popup
 */
function che_acf_filter_rest_api_preload_paths( $preload_paths ) {
	global $post;
	$rest_path    = rest_get_route_for_post( $post );
	$remove_paths = array(
		add_query_arg( 'context', 'edit', $rest_path ),
		sprintf( '%s/autosaves?context=edit', $rest_path ),
	);

	return array_filter( $preload_paths, function ( $url ) use ( $remove_paths ) {
		return ! in_array( $url, $remove_paths, true );
	} );
}

add_filter( 'block_editor_rest_api_preload_paths', 'che_acf_filter_rest_api_preload_paths', 10, 1 );

/**
 * Load json file from folder block
 */
add_filter( 'acf/settings/load_json', function ( $paths ) {
	$blocks  = che_get_all_dir_path( THEME_ACF_BLOCKS );
	$options = che_get_all_dir_path( THEME_ACF_OPTION );
	$pages   = che_get_all_dir_path( THEME_ACF_PAGES );

	return array_merge( $paths, $blocks, $options, $pages );
} );

/**
 * Save json in folder block
 * Remove file *.json on change location
 * Remove file on trash ACF group
 */
function che_save_json_in_folder_block( $path ) {

	if ( ! empty( $_POST['post_name'] ) ) {
		$files     = acf_get_local_json_files();
		$post_name = sanitize_text_field( $_POST['post_name'] );

		if ( ! empty( $files[ $post_name ] ) && file_exists( $files[ $post_name ] ) ) {
			unlink( $files[ $post_name ] );
		}
	}

	if ( ! empty( $_GET['post'] ) && ! empty( $_GET['action'] ) && $_GET['action'] == 'trash' ) {
		$post_id   = (int) $_GET['post'];
		$post_name = get_post_meta( $post_id, '_wp_desired_post_slug', true );

		che_general_search_and_delete_file( THEME_ACF_BLOCKS . '/', $post_name . '.json' );
		che_general_search_and_delete_file( THEME_ACF_PAGES . '/', $post_name . '.json' );
		che_general_search_and_delete_file( THEME_ACF_OPTION . '/', $post_name . '.json' );
		che_general_search_and_delete_file( THEME_DIR . '/acf-json/', $post_name . '.json' );
	}

	if ( isset( $_POST['acf_field_group']['location'] ) ) {

		foreach ( $_POST['acf_field_group']['location'] as $location ) {
			foreach ( $location as $rules ) {

				if ( $rules['param'] == 'block' && empty( $block_name ) ) {

					$registered_blocks = WP_Block_Type_Registry::get_instance()->get_registered( $rules['value'] );
					if ( $registered_blocks && isset( $registered_blocks->path ) && is_dir( $registered_blocks->path ) ) {
						return $registered_blocks->path;
					}

				} else if ( $rules['param'] == 'page_template' ) {
					$arr = explode( '/', $rules['value'] );

					if ( count( $arr ) === 3 && $arr[0] === 'acf-pages' && $arr[2] === 'template.php' && is_dir( THEME_ACF_PAGES . '/' . $arr[1] ) ) {
						return THEME_ACF_PAGES . '/' . $arr[1];
					}

				} else if ( $rules['param'] == 'options_page' ) {
					return THEME_ACF_OPTION . '/' . $rules['value'];
				}
			}
		}

	}

	if ( isset( $_POST['acf_ui_options_page'] ) ) {
		$dir_path = THEME_ACF_OPTION . '/' . $_POST['acf_ui_options_page']['menu_slug'];

		if ( ! is_dir( $dir_path ) ) {
			if ( ! wp_mkdir_p( $dir_path ) ) {
				error_log( "Directory $dir_path not created" );
			}
		}
	}

	return $path;
}

add_filter( 'acf/settings/save_json', 'che_save_json_in_folder_block' );

/**
 * Generate php code by field.json
 *
 * @return bool
 */
function che_generate_php_code_by_field_json(): void {
	if ( ! class_exists( 'ACFTC_Core' ) || ! class_exists( 'ACFTC_Group' ) ) {
		return;
	}

	if ( ! function_exists( 'acf_get_local_json_files' ) ) {
		return;
	}

	function che_get_post_by_name( string $name, string $post_type = "post" ) {
		$query = new WP_Query( [
			"post_type" => $post_type,
			"name"      => $name
		] );

		return $query->have_posts() ? reset( $query->posts ) : null;
	}

	function che_get_code( $key, $type ) {

		$get_post = che_get_post_by_name( $key, 'acf-field-group' );

		$field_group_post_obj = get_post( $get_post->ID );
		if ( ! $field_group_post_obj ) {
			return false;
		}

		$locations_class_name = ACFTC_Core::$class_prefix . 'Locations';
		$locations_ui         = new $locations_class_name( $field_group_post_obj );

		$parent_field_group = new ACFTC_Group( array(
			'field_group_id' => $field_group_post_obj->ID,
		) );
		$data               = $locations_ui->get_locations_code_html( $parent_field_group );

		preg_match_all( '/<code class="language-php">([^<]*)<\/code>/', $data, $matches );

		if ( ! empty( $matches[1] ) ) {
			return $matches[1];
		}

		return false;
	}

	$files = acf_get_local_json_files();

	if ( $files ) {
		foreach ( $files as $key => $value ) {

			$dir_block = THEME_ACF_BLOCKS . '/' . str_replace( 'group_block_', '', $key );
			$dir_page  = THEME_ACF_PAGES . '/' . str_replace( 'group_page_', '', $key );

			$code = false;
			$file = false;

			if ( ! empty( $dir_block ) && is_dir( $dir_block ) && ! file_exists( $dir_block . '/index.php' ) ) {
				$file     = $dir_block . '/index.php';
				$get_code = che_get_code( $key, 'block' );
				$code     = implode( PHP_EOL, $get_code );

				$code = preg_replace( '/\*\//', '*/ ' . PHP_EOL . 'if(che_show_preview_image( $block )){ return true; }', $code, 1 );
				$code = str_replace( '@param', '@var', $code );

			} elseif ( ! empty( $dir_page ) && is_dir( $dir_page ) && ! file_exists( $dir_page . '/index.php' ) ) {
				$file     = $dir_page . '/index.php';
				$get_code = che_get_code( $key, 'page' );
				$code     = implode( PHP_EOL, $get_code );
			}

			if ( $code && $file ) {
				$result = html_entity_decode( $code, ENT_QUOTES );
				file_put_contents( $file, $result );
			}

		}
	}
}

add_action( 'admin_init', 'che_generate_php_code_by_field_json' );

/**
 * Filters the $field settings array after being loaded
 *
 * The ACF field must be named `theme_icons` and has types:
 *  - 'checkbox'
 *  - 'select'
 *  - 'radio'
 *
 * @param array $field The field array containing all settings
 *
 * @return array Modified field array with SVG choices
 */
function che_acf_load_field_theme_icons( $field ): array {

	$fields_types = [
		'checkbox',
		'select',
		'radio'
	];

	// Check if the field type is one of the allowed types
	if ( ! in_array( $field['type'], $fields_types, true ) ) {
		return $field; // If not, return the field unmodified
	}

	$field['choices'] = [];
	$icons_directory  = THEME_FRONTEND_DIR . '/src/icons/';
	$file_type        = '.svg';

	$svg_files = glob( $icons_directory . '*' . $file_type );

	if ( $svg_files ) {
		foreach ( $svg_files as $svg_file ) {
			$icon_name                      = pathinfo( $svg_file, PATHINFO_FILENAME );
			$field['choices'][ $icon_name ] = $icon_name . $file_type;
		}
	}

	//set the return format and default value
	if ( ! empty( $field['choices'] ) ) {
		$field['default_value'] = reset( $field['choices'] );
		$field['return_format'] = 'value';
	}

	return $field;
}

add_filter( 'acf/load_field/name=theme_icons', 'che_acf_load_field_theme_icons' );

/**
 * Replace @year with the current year in ACF output.
 *
 * @param mixed      $value   The value to check.
 * @param int|string $post_id The post-ID.
 * @param array      $field   The field array.
 *
 * @return mixed The modified value.
 */
function che_replace_year_in_acf_output( mixed $value, int|string $post_id, array $field ): mixed {
	// Check if we are on the frontend and if the value contains @year
	if ( ! is_admin() && is_string( $value ) && str_contains( $value, '@year' ) ) {
		// Replace @year with the current year
		$value = str_replace( '@year', date( 'Y' ), $value );
	}

	return $value;
}

add_filter( 'acf/format_value', 'che_replace_year_in_acf_output', 10, 3 );

/**
 * Disable the ACF escaped HTML error messages
 *
 * @return bool
 */
add_filter( 'acf/admin/prevent_escaped_html_notice', '__return_true' );